/*
 * Copyright (c) 2014, 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.vm.ci.hotspot;

import static jdk.vm.ci.hotspot.CompilerToVM.compilerToVM;
import static jdk.vm.ci.hotspot.HotSpotJVMCIRuntime.runtime;

import java.lang.invoke.MethodHandle;

import jdk.vm.ci.common.JVMCIError;
import jdk.vm.ci.hotspot.HotSpotMethodData.VMState;
import jdk.vm.ci.meta.ConstantReflectionProvider;
import jdk.vm.ci.meta.JavaConstant;
import jdk.vm.ci.meta.MethodHandleAccessProvider;
import jdk.vm.ci.meta.ResolvedJavaField;
import jdk.vm.ci.meta.ResolvedJavaMethod;
import jdk.vm.ci.meta.ResolvedJavaType;

public class HotSpotMethodHandleAccessProvider implements MethodHandleAccessProvider {

    private final ConstantReflectionProvider constantReflection;

    public HotSpotMethodHandleAccessProvider(ConstantReflectionProvider constantReflection) {
        this.constantReflection = constantReflection;
    }

    /**
     * Lazy initialized reflection on {@link MethodHandle} internals. Field and method lookup is
     * only possible after the {@link HotSpotJVMCIRuntime} is fully initialized.
     */
    static final class Internals {
        final ResolvedJavaType lambdaFormType;
        final ResolvedJavaField methodHandleFormField;
        final ResolvedJavaField lambdaFormVmentryField;
        final HotSpotResolvedJavaField callSiteTargetField;
        final HotSpotResolvedJavaField constantCallSiteFrozenField;
        final ResolvedJavaField methodField;
        final HotSpotResolvedJavaField vmtargetField;

        /**
         * Search for an instance field with the given name in a class.
         *
         * @param declaringType the type declaring the field
         * @param fieldName name of the field to be searched
         * @param fieldType resolved Java type of the field
         * @return resolved Java field
         * @throws NoSuchFieldError
         */
        private static ResolvedJavaField findFieldInClass(ResolvedJavaType declaringType, String fieldName, ResolvedJavaType fieldType) {
            ResolvedJavaField[] fields = declaringType.getInstanceFields(false);
            for (ResolvedJavaField field : fields) {
                if (field.getName().equals(fieldName) && field.getType().equals(fieldType)) {
                    return field;
                }
            }
            throw new NoSuchFieldError(declaringType + "." + fieldName);
        }

        private static ResolvedJavaType resolveType(String className, HotSpotResolvedObjectType accessingType) {
            return (ResolvedJavaType) runtime().lookupTypeInternal(className, accessingType, true);
        }

        private Internals() {
            try {
                HotSpotResolvedObjectType accessingType = runtime().getJavaLangObject();
                ResolvedJavaType methodHandleType = resolveType("Ljava/lang/invoke/MethodHandle;", accessingType);
                ResolvedJavaType memberNameType = resolveType("Ljava/lang/invoke/MemberName;", accessingType);
                lambdaFormType = resolveType("Ljava/lang/invoke/LambdaForm;", accessingType);
                methodHandleFormField = findFieldInClass(methodHandleType, "form", lambdaFormType);
                lambdaFormVmentryField = findFieldInClass(lambdaFormType, "vmentry", memberNameType);

                ResolvedJavaType methodType = resolveType("Ljava/lang/invoke/ResolvedMethodName;", accessingType);
                methodField = findFieldInClass(memberNameType, "method", methodType);
                vmtargetField = (HotSpotResolvedJavaField) findFieldInClass(methodType, "vmtarget", resolveType(Character.toString(HotSpotJVMCIRuntime.getHostWordKind().getTypeChar()), accessingType));

                ResolvedJavaType callSiteType = resolveType("Ljava/lang/invoke/CallSite;", accessingType);
                callSiteTargetField = (HotSpotResolvedJavaField) findFieldInClass(callSiteType, "target", methodHandleType);
                ResolvedJavaType constantCallSiteType = resolveType("Ljava/lang/invoke/ConstantCallSite;", accessingType);
                ResolvedJavaType booleanType = resolveType("Z", accessingType);
                constantCallSiteFrozenField = (HotSpotResolvedJavaField) findFieldInClass(constantCallSiteType, "isFrozen", booleanType);
            } catch (Throwable ex) {
                throw new JVMCIError(ex);
            }
        }

        /**
         * Singleton instance lazily initialized via double-checked locking.
         */
        private static volatile Internals instance;

        static Internals instance() {
            Internals result = instance;
            if (result == null) {
                synchronized (VMState.class) {
                    result = instance;
                    if (result == null) {
                        instance = result = new Internals();
                    }
                }
            }
            return result;
        }

    }


    @Override
    public IntrinsicMethod lookupMethodHandleIntrinsic(ResolvedJavaMethod method) {
        int intrinsicId = ((HotSpotResolvedJavaMethodImpl) method).intrinsicId();
        if (intrinsicId != 0) {
            return getMethodHandleIntrinsic(intrinsicId);
        }
        return null;
    }

    public static IntrinsicMethod getMethodHandleIntrinsic(int intrinsicId) {
        HotSpotVMConfig config = runtime().getConfig();
        if (intrinsicId == config.vmIntrinsicInvokeBasic) {
            return IntrinsicMethod.INVOKE_BASIC;
        } else if (intrinsicId == config.vmIntrinsicLinkToInterface) {
            return IntrinsicMethod.LINK_TO_INTERFACE;
        } else if (intrinsicId == config.vmIntrinsicLinkToSpecial) {
            return IntrinsicMethod.LINK_TO_SPECIAL;
        } else if (intrinsicId == config.vmIntrinsicLinkToStatic) {
            return IntrinsicMethod.LINK_TO_STATIC;
        } else if (intrinsicId == config.vmIntrinsicLinkToVirtual) {
            return IntrinsicMethod.LINK_TO_VIRTUAL;
        } else if (intrinsicId == config.vmIntrinsicLinkToNative) {
            return IntrinsicMethod.LINK_TO_NATIVE;
        }
        return null;
    }

    @Override
    public ResolvedJavaMethod resolveInvokeBasicTarget(JavaConstant methodHandle, boolean forceBytecodeGeneration) {
        if (methodHandle.isNull()) {
            return null;
        }

        /* Load non-public field: LambdaForm MethodHandle.form */
        Internals internals = Internals.instance();
        JavaConstant lambdaForm = constantReflection.readFieldValue(internals.methodHandleFormField, methodHandle);
        if (lambdaForm == null || lambdaForm.isNull()) {
            return null;
        }

        JavaConstant memberName = constantReflection.readFieldValue(internals.lambdaFormVmentryField, lambdaForm);
        if (memberName.isNull() && forceBytecodeGeneration) {
            compilerToVM().compileToBytecode((HotSpotObjectConstantImpl) lambdaForm);
            memberName = constantReflection.readFieldValue(internals.lambdaFormVmentryField, lambdaForm);
            assert memberName.isNonNull();
        }
        JavaConstant method = constantReflection.readFieldValue(internals.methodField, memberName);
        return getTargetMethod(method);
    }

    @Override
    public ResolvedJavaMethod resolveLinkToTarget(JavaConstant memberName) {
        if (memberName.isNull()) {
            return null;
        }
        JavaConstant method = constantReflection.readFieldValue(Internals.instance().methodField, memberName);
        return getTargetMethod(method);
    }

    /**
     * Returns the {@link ResolvedJavaMethod} for the method of a java.lang.invoke.MemberName.
     */
    private static ResolvedJavaMethod getTargetMethod(JavaConstant method) {
        if (method == null) {
            // If readFieldValue returns NULL the type was wrong
            throw new IllegalArgumentException("unexpected type for memberName");
        }

        /* Read the ResolvedJavaMethod from the injected field MemberName.method.vmtarget */
        return compilerToVM().getResolvedJavaMethod((HotSpotObjectConstantImpl) method, Internals.instance().vmtargetField.getOffset());
    }
}
